/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.cloudera.com.amazonaws.AmazonClientException;
import com.cloudera.com.amazonaws.AmazonServiceException;
import com.cloudera.com.amazonaws.auth.AWSCredentialsProvider;
import com.cloudera.com.amazonaws.auth.EnvironmentVariableCredentialsProvider;
import com.cloudera.com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.cloudera.com.amazonaws.services.s3.model.AmazonS3Exception;
import com.cloudera.com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Preconditions;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.file.AccessDeniedException;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.s3a.AWSClientIOException;
import org.apache.hadoop.fs.s3a.AWSCredentialProviderList;
import org.apache.hadoop.fs.s3a.AWSS3IOException;
import org.apache.hadoop.fs.s3a.AWSServiceIOException;
import org.apache.hadoop.fs.s3a.BasicAWSCredentialsProvider;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3native.S3xLoginHelper;
import org.apache.hadoop.security.ProviderUtils;
import org.slf4j.Logger;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public final class S3AUtils {
    private static final Logger LOG = S3AFileSystem.LOG;
    static final String CONSTRUCTOR_EXCEPTION = "constructor exception";
    static final String INSTANTIATION_EXCEPTION = "instantiation exception";
    static final String NOT_AWS_PROVIDER = "does not implement AWSCredentialsProvider";

    private S3AUtils() {
    }

    public static IOException translateException(String operation, Path path, AmazonClientException exception) {
        return S3AUtils.translateException(operation, path.toString(), exception);
    }

    public static IOException translateException(String operation, String path, AmazonClientException exception) {
        IOException ioe;
        String message = String.format("%s%s: %s", operation, path != null ? " on " + path : "", exception);
        if (!(exception instanceof AmazonServiceException)) {
            return new AWSClientIOException(message, exception);
        }
        AmazonServiceException ase = (AmazonServiceException)exception;
        AmazonS3Exception s3Exception = ase instanceof AmazonS3Exception ? (AmazonS3Exception)ase : null;
        int status = ase.getStatusCode();
        switch (status) {
            case 401: 
            case 403: {
                ioe = new AccessDeniedException(path, null, message);
                ioe.initCause(ase);
                break;
            }
            case 404: 
            case 410: {
                ioe = new FileNotFoundException(message);
                ioe.initCause(ase);
                break;
            }
            case 416: {
                ioe = new EOFException(message);
                break;
            }
            default: {
                ioe = s3Exception != null ? new AWSS3IOException(message, s3Exception) : new AWSServiceIOException(message, ase);
            }
        }
        return ioe;
    }

    public static IOException extractException(String operation, String path, ExecutionException ee) {
        Throwable cause = ee.getCause();
        IOException ioe = cause instanceof AmazonClientException ? S3AUtils.translateException(operation, path, (AmazonClientException)cause) : (cause instanceof IOException ? (IOException)cause : new IOException(operation + " failed: " + cause, cause));
        return ioe;
    }

    public static String stringify(AmazonServiceException e) {
        StringBuilder builder = new StringBuilder(String.format("%s: %s error %d: %s; %s%s%n", new Object[]{e.getErrorType(), e.getServiceName(), e.getStatusCode(), e.getErrorCode(), e.getErrorMessage(), e.isRetryable() ? " (retryable)" : ""}));
        String rawResponseContent = e.getRawResponseContent();
        if (rawResponseContent != null) {
            builder.append(rawResponseContent);
        }
        return builder.toString();
    }

    public static String stringify(AmazonS3Exception e) {
        StringBuilder builder = new StringBuilder(S3AUtils.stringify((AmazonServiceException)e));
        Map<String, String> details = e.getAdditionalDetails();
        if (details != null) {
            builder.append('\n');
            for (Map.Entry<String, String> d : details.entrySet()) {
                builder.append(d.getKey()).append('=').append(d.getValue()).append('\n');
            }
        }
        return builder.toString();
    }

    public static S3AFileStatus createFileStatus(Path keyPath, S3ObjectSummary summary, long blockSize) {
        if (S3AUtils.objectRepresentsDirectory(summary.getKey(), summary.getSize())) {
            return new S3AFileStatus(true, true, keyPath);
        }
        return new S3AFileStatus(summary.getSize(), S3AUtils.dateToLong(summary.getLastModified()), keyPath, blockSize);
    }

    public static boolean objectRepresentsDirectory(String name, long size) {
        return !name.isEmpty() && name.charAt(name.length() - 1) == '/' && size == 0L;
    }

    public static long dateToLong(Date date) {
        if (date == null) {
            return 0L;
        }
        return date.getTime();
    }

    public static AWSCredentialProviderList createAWSCredentialProviderSet(URI binding, Configuration conf, URI fsURI) throws IOException {
        Class[] awsClasses;
        AWSCredentialProviderList credentials = new AWSCredentialProviderList();
        try {
            awsClasses = conf.getClasses("fs.s3a.aws.credentials.provider", new Class[0]);
        }
        catch (RuntimeException e) {
            Throwable c = e.getCause() != null ? e.getCause() : e;
            throw new IOException("From option fs.s3a.aws.credentials.provider " + c, c);
        }
        if (awsClasses.length == 0) {
            S3xLoginHelper.Login creds = S3AUtils.getAWSAccessKeys(binding, conf);
            credentials.add(new BasicAWSCredentialsProvider(creds.getUser(), creds.getPassword()));
            credentials.add(new EnvironmentVariableCredentialsProvider());
            credentials.add(new InstanceProfileCredentialsProvider());
        } else {
            for (Class aClass : awsClasses) {
                credentials.add(S3AUtils.createAWSCredentialProvider(conf, aClass, fsURI));
            }
        }
        return credentials;
    }

    static AWSCredentialsProvider createAWSCredentialProvider(Configuration conf, Class<?> credClass, URI uri) throws IOException {
        AWSCredentialsProvider credentials;
        String className = credClass.getName();
        if (!AWSCredentialsProvider.class.isAssignableFrom(credClass)) {
            throw new IOException("Class " + credClass + " " + NOT_AWS_PROVIDER);
        }
        try {
            LOG.debug("Credential provider class is {}", (Object)className);
            try {
                credentials = (AWSCredentialsProvider)credClass.getDeclaredConstructor(URI.class, Configuration.class).newInstance(uri, conf);
            }
            catch (NoSuchMethodException | SecurityException e) {
                credentials = (AWSCredentialsProvider)credClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
        }
        catch (NoSuchMethodException | SecurityException e) {
            throw new IOException(String.format("%s constructor exception.  A class specified in %s must provide an accessible constructor accepting URI and Configuration, or an accessible default constructor.", className, "fs.s3a.aws.credentials.provider"), e);
        }
        catch (IllegalArgumentException | ReflectiveOperationException e) {
            throw new IOException(className + " " + INSTANTIATION_EXCEPTION + ".", e);
        }
        LOG.debug("Using {} for {}.", (Object)credentials, (Object)uri);
        return credentials;
    }

    public static S3xLoginHelper.Login getAWSAccessKeys(URI name, Configuration conf) throws IOException {
        S3xLoginHelper.Login login = S3xLoginHelper.extractLoginDetailsWithWarnings(name);
        Configuration c = ProviderUtils.excludeIncompatibleCredentialProviders((Configuration)conf, S3AFileSystem.class);
        String accessKey = S3AUtils.getPasswordLegacy(c, "fs.s3a.access.key", login.getUser());
        String secretKey = S3AUtils.getPasswordLegacy(c, "fs.s3a.secret.key", login.getPassword());
        return new S3xLoginHelper.Login(accessKey, secretKey);
    }

    private static String getPasswordLegacy(Configuration conf, String key, String val) throws IOException {
        String v = S3AUtils.getPassword(conf, key, val);
        if (v.equals("")) {
            if (key.equals("fs.s3a.access.key")) {
                return S3AUtils.getPassword(conf, "fs.s3a.awsAccessKeyId", val);
            }
            if (key.equals("fs.s3a.secret.key")) {
                return S3AUtils.getPassword(conf, "fs.s3a.awsSecretAccessKey", val);
            }
        }
        return v;
    }

    static String getPassword(Configuration conf, String key, String val) throws IOException {
        return StringUtils.isEmpty((String)val) ? S3AUtils.lookupPassword(conf, key, "") : val;
    }

    static String lookupPassword(Configuration conf, String key, String defVal) throws IOException {
        try {
            char[] pass = conf.getPassword(key);
            return pass != null ? new String(pass).trim() : defVal;
        }
        catch (IOException ioe) {
            throw new IOException("Cannot find password option " + key, ioe);
        }
    }

    public static String stringify(S3ObjectSummary summary) {
        StringBuilder builder = new StringBuilder(summary.getKey().length() + 100);
        builder.append(summary.getKey()).append(' ');
        builder.append("size=").append(summary.getSize());
        return builder.toString();
    }

    static int intOption(Configuration conf, String key, int defVal, int min) {
        int v = conf.getInt(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        return v;
    }

    static long longOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLong(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        return v;
    }
}

