/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.registry.client.impl.zk;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.security.auth.login.AppConfigurationEntry;
import org.apache.commons.lang.StringUtils;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.authentication.util.KerberosUtil;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.service.ServiceStateException;
import org.apache.hadoop.util.ZKUtil;
import org.apache.zookeeper.Environment;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Id;
import org.apache.zookeeper.server.auth.DigestAuthenticationProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RegistrySecurity
extends AbstractService {
    private static final Logger LOG = LoggerFactory.getLogger(RegistrySecurity.class);
    public static final String E_UNKNOWN_AUTHENTICATION_MECHANISM = "Unknown/unsupported authentication mechanism; ";
    public static final String E_NO_USER_DETERMINED_FOR_ACLS = "No user for ACLs determinable from current user or registry option hadoop.registry.user.accounts";
    public static final String E_NO_KERBEROS = "Registry security is enabled -but Hadoop security is not enabled";
    private AccessPolicy access;
    private String digestAuthUser;
    private String digestAuthPassword;
    private byte[] digestAuthData;
    private boolean secureRegistry;
    public static final ACL ALL_READWRITE_ACCESS = new ACL(31, ZooDefs.Ids.ANYONE_ID_UNSAFE);
    public static final ACL ALL_READ_ACCESS = new ACL(1, ZooDefs.Ids.ANYONE_ID_UNSAFE);
    public static final List<ACL> WorldReadWriteACL;
    private final List<ACL> systemACLs = new ArrayList<ACL>();
    private final List<ACL> digestACLs = new ArrayList<ACL>();
    private String kerberosRealm;
    private String jaasClientContext;
    private String jaasClientIdentity;
    private static final String JAAS_ENTRY = "%s { %n %s required%n keyTab=\"%s\"%n debug=true%n principal=\"%s\"%n useKeyTab=true%n useTicketCache=false%n doNotPrompt=true%n storeKey=true;%n}; %n";

    public RegistrySecurity(String name) {
        super(name);
    }

    protected void serviceInit(Configuration conf) throws Exception {
        String auth;
        super.serviceInit(conf);
        switch (auth = conf.getTrimmed("hadoop.registry.client.auth", "")) {
            case "kerberos": {
                this.access = AccessPolicy.sasl;
                break;
            }
            case "digest": {
                this.access = AccessPolicy.digest;
                break;
            }
            case "": {
                this.access = AccessPolicy.anon;
                break;
            }
            default: {
                throw new ServiceStateException("Unknown/unsupported authentication mechanism; \"" + auth + "\"");
            }
        }
        this.initSecurity();
    }

    private void initSecurity() throws IOException {
        this.secureRegistry = this.getConfig().getBoolean("hadoop.registry.secure", false);
        this.systemACLs.clear();
        if (this.secureRegistry) {
            ACL self;
            this.addSystemACL(ALL_READ_ACCESS);
            this.kerberosRealm = this.getConfig().get("hadoop.registry.kerberos.realm", RegistrySecurity.getDefaultRealmInJVM());
            String system = this.getOrFail("hadoop.registry.system.accounts", "sasl:yarn@, sasl:mapred@, sasl:hdfs@, sasl:hadoop@");
            this.systemACLs.addAll(this.buildACLs(system, this.kerberosRealm, 31));
            String user = this.getConfig().get("hadoop.registry.user.accounts", "");
            List<ACL> userACLs = this.buildACLs(user, this.kerberosRealm, 31);
            if (UserGroupInformation.isSecurityEnabled() && (self = this.createSaslACLFromCurrentUser(31)) != null) {
                userACLs.add(self);
            }
            switch (this.access) {
                case sasl: {
                    if (!UserGroupInformation.isSecurityEnabled()) {
                        throw new IOException("Kerberos required for secure registry access");
                    }
                    UserGroupInformation currentUser = UserGroupInformation.getCurrentUser();
                    this.jaasClientContext = this.getOrFail("hadoop.registry.jaas.context", "Client");
                    this.jaasClientIdentity = currentUser.getShortUserName();
                    if (!LOG.isDebugEnabled()) break;
                    LOG.debug("Auth is SASL user=\"{}\" JAAS context=\"{}\"", (Object)this.jaasClientIdentity, (Object)this.jaasClientContext);
                    break;
                }
                case digest: {
                    String id = this.getOrFail("hadoop.registry.client.auth.id", "");
                    String pass = this.getOrFail("hadoop.registry.client.auth.password", "");
                    if (userACLs.isEmpty()) {
                        throw new ServiceStateException(E_NO_USER_DETERMINED_FOR_ACLS);
                    }
                    this.digest(id, pass);
                    ACL acl = new ACL(31, this.toDigestId(id, pass));
                    userACLs.add(acl);
                    this.digestAuthUser = id;
                    this.digestAuthPassword = pass;
                    String authPair = id + ":" + pass;
                    this.digestAuthData = authPair.getBytes("UTF-8");
                    if (!LOG.isDebugEnabled()) break;
                    LOG.debug("Auth is Digest ACL: {}", (Object)RegistrySecurity.aclToString(acl));
                    break;
                }
                case anon: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Auth is anonymous");
                    }
                    userACLs = new ArrayList<ACL>(0);
                }
            }
            this.systemACLs.addAll(userACLs);
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Registry has no security");
            }
            this.systemACLs.addAll(WorldReadWriteACL);
        }
    }

    public void addSystemACL(ACL acl) {
        this.systemACLs.add(acl);
    }

    public boolean addDigestACL(ACL acl) {
        if (this.secureRegistry) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Added ACL {}", (Object)RegistrySecurity.aclToString(acl));
            }
            this.digestACLs.add(acl);
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignoring added ACL - registry is insecure{}", (Object)RegistrySecurity.aclToString(acl));
        }
        return false;
    }

    public void resetDigestACLs() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Cleared digest ACLs");
        }
        this.digestACLs.clear();
    }

    public boolean isSecureRegistry() {
        return this.secureRegistry;
    }

    public List<ACL> getSystemACLs() {
        Preconditions.checkNotNull(this.systemACLs, (Object)"registry security is unitialized");
        return Collections.unmodifiableList(this.systemACLs);
    }

    public List<ACL> getClientACLs() {
        ArrayList<ACL> clientACLs = new ArrayList<ACL>(this.systemACLs);
        clientACLs.addAll(this.digestACLs);
        return clientACLs;
    }

    public ACL createSaslACLFromCurrentUser(int perms) throws IOException {
        UserGroupInformation currentUser = UserGroupInformation.getCurrentUser();
        if (currentUser.hasKerberosCredentials()) {
            return this.createSaslACL(currentUser, perms);
        }
        return null;
    }

    public ACL createSaslACL(UserGroupInformation ugi, int perms) {
        String userName = ugi.getUserName();
        return new ACL(perms, new Id("sasl", userName));
    }

    private String getOrFail(String key, String defval) throws IOException {
        String val = this.getConfig().get(key, defval);
        if (StringUtils.isEmpty((String)val)) {
            throw new IOException("Missing value for configuration option " + key);
        }
        return val;
    }

    public boolean isValid(String idPasswordPair) {
        String[] parts = idPasswordPair.split(":");
        return parts.length == 2 && !StringUtils.isEmpty((String)parts[0]) && !StringUtils.isEmpty((String)parts[1]);
    }

    public String getKerberosRealm() {
        return this.kerberosRealm;
    }

    public String digest(String idPasswordPair) throws IOException {
        if (StringUtils.isEmpty((String)idPasswordPair) || !this.isValid(idPasswordPair)) {
            throw new IOException("Invalid id:password: " + idPasswordPair);
        }
        try {
            return DigestAuthenticationProvider.generateDigest((String)idPasswordPair);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException(e.toString(), e);
        }
    }

    public String digest(String id, String password) throws IOException {
        return this.digest(id + ":" + password);
    }

    public Id toDigestId(String digest) {
        return new Id("digest", digest);
    }

    public Id toDigestId(String id, String password) throws IOException {
        return this.toDigestId(this.digest(id, password));
    }

    public List<String> splitAclPairs(String aclString, String realm) {
        ArrayList list = Lists.newArrayList((Iterable)Splitter.on((char)',').omitEmptyStrings().trimResults().split((CharSequence)aclString));
        ListIterator<String> listIterator = list.listIterator();
        while (listIterator.hasNext()) {
            String next = (String)listIterator.next();
            if (!next.startsWith("sasl:") || !next.endsWith("@")) continue;
            listIterator.set(next + realm);
        }
        return list;
    }

    public Id parse(String idPair, String realm) {
        int firstColon = idPair.indexOf(58);
        int lastColon = idPair.lastIndexOf(58);
        if (firstColon == -1 || lastColon == -1 || firstColon != lastColon) {
            throw new IllegalArgumentException("ACL '" + idPair + "' not of expected form scheme:id");
        }
        String scheme = idPair.substring(0, firstColon);
        String id = idPair.substring(firstColon + 1);
        if (id.endsWith("@")) {
            Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)realm), (String)"@ suffixed account but no realm %s", (Object[])new Object[]{id});
            id = id + realm;
        }
        return new Id(scheme, id);
    }

    public List<ACL> buildACLs(String principalList, String realm, int perms) throws IOException {
        List<String> aclPairs = this.splitAclPairs(principalList, realm);
        ArrayList<ACL> ids = new ArrayList<ACL>(aclPairs.size());
        for (String aclPair : aclPairs) {
            ACL newAcl = new ACL();
            newAcl.setId(this.parse(aclPair, realm));
            newAcl.setPerms(perms);
            ids.add(newAcl);
        }
        return ids;
    }

    public List<ACL> parseACLs(String zkAclConf) throws IOException {
        try {
            return ZKUtil.parseACLs((String)ZKUtil.resolveConfIndirection((String)zkAclConf));
        }
        catch (ZKUtil.BadAclFormatException e) {
            throw new IOException("Parsing " + zkAclConf + " :" + (Object)((Object)e), e);
        }
    }

    public static String getKerberosAuthModuleForJVM() {
        if (System.getProperty("java.vendor").contains("IBM")) {
            return "com.ibm.security.auth.module.Krb5LoginModule";
        }
        return "com.sun.security.auth.module.Krb5LoginModule";
    }

    public String createJAASEntry(String context, String principal, File keytab) {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)principal), (Object)"invalid principal");
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)context), (Object)"invalid context");
        Preconditions.checkArgument((keytab != null && keytab.isFile() ? 1 : 0) != 0, (Object)"Keytab null or missing: ");
        String keytabpath = keytab.getAbsolutePath();
        keytabpath = keytabpath.replace('\\', '/');
        return String.format(Locale.ENGLISH, JAAS_ENTRY, context, RegistrySecurity.getKerberosAuthModuleForJVM(), keytabpath, principal);
    }

    public static void bindJVMtoJAASFile(File jaasFile) {
        String path = jaasFile.getAbsolutePath();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Binding {} to {}", (Object)Environment.JAAS_CONF_KEY, (Object)path);
        }
        System.setProperty(Environment.JAAS_CONF_KEY, path);
    }

    public static void bindZKToServerJAASContext(String contextName) {
        System.setProperty("zookeeper.sasl.serverconfig", contextName);
    }

    public static void clearJaasSystemProperties() {
        System.clearProperty(Environment.JAAS_CONF_KEY);
    }

    public static AppConfigurationEntry[] validateContext(String context) {
        if (context == null) {
            throw new RuntimeException("Null context argument");
        }
        if (context.isEmpty()) {
            throw new RuntimeException("Empty context argument");
        }
        javax.security.auth.login.Configuration configuration = javax.security.auth.login.Configuration.getConfiguration();
        AppConfigurationEntry[] entries = configuration.getAppConfigurationEntry(context);
        if (entries == null) {
            throw new RuntimeException(String.format("Entry \"%s\" not found; JAAS config = %s", context, RegistrySecurity.describeProperty(Environment.JAAS_CONF_KEY)));
        }
        return entries;
    }

    public void applySecurityEnvironment(CuratorFrameworkFactory.Builder builder) {
        if (this.isSecureRegistry()) {
            switch (this.access) {
                case anon: {
                    RegistrySecurity.clearZKSaslClientProperties();
                    break;
                }
                case digest: {
                    RegistrySecurity.clearZKSaslClientProperties();
                    builder.authorization("digest", this.digestAuthData);
                    break;
                }
                case sasl: {
                    RegistrySecurity.setZKSaslClientProperties(this.jaasClientIdentity, this.jaasClientContext);
                }
            }
        }
    }

    public static void setZKSaslClientProperties(String username, String context) {
        RegistrySecurity.validateContext(context);
        RegistrySecurity.enableZookeeperClientSASL();
        System.setProperty("zookeeper.sasl.client.username", username);
        System.setProperty("zookeeper.sasl.clientconfig", context);
    }

    public static void clearZKSaslClientProperties() {
        RegistrySecurity.disableZookeeperClientSASL();
        System.clearProperty("zookeeper.sasl.clientconfig");
        System.clearProperty("zookeeper.sasl.client.username");
    }

    protected static void enableZookeeperClientSASL() {
        System.setProperty("zookeeper.sasl.client", "true");
    }

    public static void disableZookeeperClientSASL() {
        System.setProperty("zookeeper.sasl.client", "false");
    }

    public static boolean isClientSASLEnabled() {
        return Boolean.valueOf(System.getProperty("zookeeper.sasl.client", "true"));
    }

    public void logCurrentHadoopUser() {
        try {
            UserGroupInformation currentUser = UserGroupInformation.getCurrentUser();
            LOG.info("Current user = {}", (Object)currentUser);
            UserGroupInformation realUser = currentUser.getRealUser();
            LOG.info("Real User = {}", (Object)realUser);
        }
        catch (IOException e) {
            LOG.warn("Failed to get current user {}, {}", (Throwable)e);
        }
    }

    public static String aclsToString(List<ACL> acls) {
        StringBuilder builder = new StringBuilder();
        if (acls == null) {
            builder.append("null ACL");
        } else {
            builder.append('\n');
            for (ACL acl : acls) {
                builder.append(RegistrySecurity.aclToString(acl)).append(" ");
            }
        }
        return builder.toString();
    }

    public static String aclToString(ACL acl) {
        return String.format(Locale.ENGLISH, "0x%02x: %s", acl.getPerms(), RegistrySecurity.idToString(acl.getId()));
    }

    public static String idToString(Id id) {
        String s;
        if (id.getScheme().equals("digest")) {
            String ids = id.getId();
            int colon = ids.indexOf(58);
            if (colon > 0) {
                ids = ids.substring(colon + 3);
            }
            s = "digest: " + ids;
        } else {
            s = id.toString();
        }
        return s;
    }

    public String buildSecurityDiagnostics() {
        StringBuilder builder = new StringBuilder();
        builder.append(this.secureRegistry ? "secure registry; " : "insecure registry; ");
        builder.append("Curator service access policy: ").append((Object)this.access);
        builder.append("; System ACLs: ").append(RegistrySecurity.aclsToString(this.systemACLs));
        builder.append("User: ").append(UgiInfo.fromCurrentUser());
        builder.append("; Kerberos Realm: ").append(this.kerberosRealm);
        builder.append(RegistrySecurity.describeProperty(Environment.JAAS_CONF_KEY));
        String sasl = System.getProperty("zookeeper.sasl.client", "true");
        boolean saslEnabled = Boolean.valueOf(sasl);
        builder.append(RegistrySecurity.describeProperty("zookeeper.sasl.client", "true"));
        if (saslEnabled) {
            builder.append("; JAAS Client Identity").append("=").append(this.jaasClientIdentity).append("; ");
            builder.append("hadoop.registry.jaas.context").append("=").append(this.jaasClientContext).append("; ");
            builder.append(RegistrySecurity.describeProperty("zookeeper.sasl.client.username"));
            builder.append(RegistrySecurity.describeProperty("zookeeper.sasl.clientconfig"));
        }
        builder.append(RegistrySecurity.describeProperty("zookeeper.allowSaslFailedClients", "(undefined but defaults to true)"));
        builder.append(RegistrySecurity.describeProperty("zookeeper.maintain_connection_despite_sasl_failure"));
        return builder.toString();
    }

    private static String describeProperty(String name) {
        return RegistrySecurity.describeProperty(name, "(undefined)");
    }

    private static String describeProperty(String name, String def) {
        return "; " + name + "=" + System.getProperty(name, def);
    }

    public static String getDefaultRealmInJVM() {
        try {
            return KerberosUtil.getDefaultRealm();
        }
        catch (ClassNotFoundException ignored) {
        }
        catch (NoSuchMethodException ignored) {
        }
        catch (IllegalAccessException ignored) {
        }
        catch (InvocationTargetException invocationTargetException) {
            // empty catch block
        }
        return "";
    }

    public ACL createACLForUser(UserGroupInformation ugi, int perms) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Creating ACL For ", (Object)new UgiInfo(ugi));
        }
        if (!this.secureRegistry) {
            return ALL_READWRITE_ACCESS;
        }
        return this.createACLfromUsername(ugi.getUserName(), perms);
    }

    public ACL createACLfromUsername(String username, int perms) {
        if (!username.contains("@")) {
            username = username + "@" + this.kerberosRealm;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Appending kerberos realm to make {}", (Object)username);
            }
        }
        return new ACL(perms, new Id("sasl", username));
    }

    static {
        ArrayList<ACL> acls = new ArrayList<ACL>();
        acls.add(ALL_READWRITE_ACCESS);
        WorldReadWriteACL = new CopyOnWriteArrayList<ACL>(acls);
    }

    public static class AclListInfo {
        public final List<ACL> acls;

        public AclListInfo(List<ACL> acls) {
            this.acls = acls;
        }

        public String toString() {
            return RegistrySecurity.aclsToString(this.acls);
        }
    }

    public static class UgiInfo {
        private final UserGroupInformation ugi;

        public static UgiInfo fromCurrentUser() {
            try {
                return new UgiInfo(UserGroupInformation.getCurrentUser());
            }
            catch (IOException e) {
                LOG.info("Failed to get current user {}", (Object)e, (Object)e);
                return new UgiInfo(null);
            }
        }

        public UgiInfo(UserGroupInformation ugi) {
            this.ugi = ugi;
        }

        public String toString() {
            if (this.ugi == null) {
                return "(null ugi)";
            }
            StringBuilder builder = new StringBuilder();
            builder.append(this.ugi.getUserName()).append(": ");
            builder.append(this.ugi.toString());
            builder.append(" hasKerberosCredentials=").append(this.ugi.hasKerberosCredentials());
            builder.append(" isFromKeytab=").append(this.ugi.isFromKeytab());
            builder.append(" kerberos is enabled in Hadoop =").append(UserGroupInformation.isSecurityEnabled());
            return builder.toString();
        }
    }

    private static enum AccessPolicy {
        anon,
        sasl,
        digest;

    }
}

